function approxhess = getApproxHessian(problem, x, d, storedb, key)
% Computes an approximation of the Hessian of the cost fun. at x along d.
%
% function approxhess = getApproxHessian(problem, x, d)
% function approxhess = getApproxHessian(problem, x, d, storedb)
% function approxhess = getApproxHessian(problem, x, d, storedb, key)
%
% Returns an approximation of the Hessian at x along d of the cost function
% described in the problem structure.
%
% storedb is a StoreDB object, key is the StoreDB key to point x.
%
% If no approximate Hessian was provided, this call is redirected to
% getHessianFD.
% 
% See also: getHessianFD canGetApproxHessian

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Dec. 30, 2012.
% Contributors: 
% Change log: 
%
%   April 3, 2015 (NB):
%       Works with the new StoreDB class system.

    % Allow omission of the key, and even of storedb.
    if ~exist('key', 'var')
        if ~exist('storedb', 'var')
            storedb = StoreDB();
        end
        key = storedb.getNewKey();
    end


    if isfield(problem, 'approxhess')
    %% Compute the approximate Hessian using approxhess.
        
        % Check whether this function wants to deal with storedb or not.
        switch nargin(problem.approxhess)
            case 2
                approxhess = problem.approxhess(x, d);
            case 3
                % Obtain, pass along, and save the store for x.
                store = storedb.getWithShared(key);
                [approxhess, store] = problem.approxhess(x, d, store);
                storedb.setWithShared(store, key);
            case 4
                % Pass along the whole storedb (by reference), with key.
                approxhess = problem.approxhess(x, d, storedb, key);
            otherwise
                up = MException('manopt:getApproxHessian:badapproxhess', ...
                    'approxhess should accept 2, 3 or 4 inputs.');
                throw(up);
        end
        
    else
    %% Try to fall back to a standard FD approximation.
    
        approxhess = getHessianFD(problem, x, d, storedb, key);
        
    end
    
end
